/**
 * \file: grl_bitmap_decoder_core.h
 *
 * \version: $Id: grl_bitmap_decoder_core.h,v 1.17 2010/01/14 13:43:29 tkniep Exp $ 
 *
 * This header file declares the internal functions of the 
 * Bitmap Decoder
 *
 * \component: SVG Bitmap Decoder (SVGBMPDEC)
 *
 * \author: T. Kniep (tkniep@de.adit-jv.com)
 *
 * \copyright: (c) 2009 ADIT Corporation
 *
 ***********************************************************************/
 
#ifndef GRL_BITMAP_DECODER_CORE_H
#define GRL_BITMAP_DECODER_CORE_H

#define SHARED_LIB

/** Check and remember the unit type */
#if ((defined SYSPROG) || (defined RESIDENT_UNIT))
# define GRL_BMPDEC_KERNEL_MODULE
#elif (defined SHARED_LIB)
# define GRL_BMPDEC_SHARED_LIB
#else
# error "Unsupported configuration. Must be SYSPROG, RESIDENT_UNIT or SHARED_LIB."
#endif 


#ifdef GRL_BMPDEC_KERNEL_MODULE

#elif defined GRL_BMPDEC_SHARED_LIB
//#  include <extension/outer.h>
#endif

#ifndef BUILDENV_LINUX
#	include <util_lib.h>
#endif

#include "svg_bitmap_decoder.h"

/*******************************************************************************************
 *   Macro definitions
 *******************************************************************************************/

/* 
 * Resolver keys for looking up the decoder functions.
 */
#define GRL_BMPDEC_FUNC_DUMMY           ((signed char*)(void*)"")
#define GRL_BMPDEC_FUNC_DRAW_PNG        ((signed char*)(void*)"ADSVGBMPDEC_DRAW_PNG")
#define GRL_BMPDEC_FUNC_INFO_PNG        ((signed char*)(void*)"ADSVGBMPDEC_INFO_PNG")
#define GRL_BMPDEC_FUNC_TYPE_PNG        ((signed char*)(void*)"ADSVGBMPDEC_TYPE_PNG")
#define GRL_BMPDEC_FUNC_DRAW_JPEG       ((signed char*)(void*)"ADSVGBMPDEC_DRAW_JPEG")
#define GRL_BMPDEC_FUNC_INFO_JPEG       ((signed char*)(void*)"ADSVGBMPDEC_INFO_JPEG")
#define GRL_BMPDEC_FUNC_TYPE_JPEG       ((signed char*)(void*)"ADSVGBMPDEC_TYPE_JPEG")

#define GRL_BMPDEC_HUNDRED              (100)
#define GRL_BMPDEC_TEN                  (10)

/* Position indices in dsnames */
#define GRL_BMPDEC_DSNM_6               (6)     /**< Second last char */
#define GRL_BMPDEC_DSNM_7               (7)     /**< Last char        */

/** Char 0 for computing dsnames */
#define GRL_BMPDEC_CHAR_0               '0'

/* 
 * Flag patterns for synchronization between API and decoder tasks 
 * Bits  0-15:       API -> Dec. Task
 * Bits 16-31: Dec. Task -> API
 */
#define GRL_BMPDEC_FLG_PTN_START        (SVGUint32)(0x00000001)
#define GRL_BMPDEC_FLG_PTN_ABORT        (SVGUint32)(0x00000010)
#define GRL_BMPDEC_FLG_PTN_FINISHED     (SVGUint32)(0x00010000)

/* Unit type dependent RTOS function calls */
#ifdef GRL_BMPDEC_KERNEL_MODULE
# define GRL_BMPDEC_CRE_FLG( a, b, c )          \
    do {                                        \
        (a).flgatr = ((a).flgatr | TA_DSNAME);  \
        strncpy( (void*)(a).dsname,             \
                 (c),                           \
                 GRL_BMPDEC_DSNAME_LEN );       \
        (b) = tk_cre_flg( &(a) );               \
    } while (0)


# define GRL_BMPDEC_WAI_FLG( a, b, c, d)        tk_wai_flg( ctx->dec_flg_id, a, b, c, d)

# define GRL_BMPDEC_SET_FLG( a )                tk_set_flg( ctx->dec_flg_id, a )

# define GRL_BMPDEC_CLR_FLG( a )                tk_clr_flg( ctx->dec_flg_id, a )

# define GRL_BMPDEC_TER_TSK( a, b )     \
    do {                                \
        tk_ter_tsk( a );                \
        (b) = tk_del_tsk( a );          \
} while (0)

# define GRL_BMPDEC_EXT_TSK                     tk_exd_tsk()
    

#elif defined GRL_BMPDEC_SHARED_LIB

#ifdef BUILDENV_LINUX

# define GRL_BMPDEC_CRE_FLG( a, b )       	((b) = GRL_cre_flg( &(a)))

# define GRL_BMPDEC_WAI_FLG( a, b, c, d )       GRL_wai_flg( ctx->dec_flg , a, b, c , d )

# define GRL_BMPDEC_SET_FLG( a )                GRL_set_flg( ctx->dec_flg , a )

# define GRL_BMPDEC_CLR_FLG( a )                GRL_clr_flg( ctx->dec_flg , a )

# define GRL_BMPDEC_DEL_FLG( a )		GRL_del_flg( a )

# define GRL_BMPDEC_CRS_TSK( a , b , c, d)	pthread_create( a, b , c , d);

# define GRL_BMPDEC_TER_TSK( a, b )             ((b) = pthread_cancel( a ))

# define GRL_BMPDEC_EXT_TSK                     pthread_exit( NULL );

#else
# define GRL_BMPDEC_CRE_FLG( a, b, c )          (b) = tkse_cre_flg( &(a) )

# define GRL_BMPDEC_WAI_FLG( a, b, c, d)        tkse_wai_flg( ctx->dec_flg_id, a, b, c, d)

# define GRL_BMPDEC_SET_FLG( a )                tkse_set_flg( ctx->dec_flg_id, a )

# define GRL_BMPDEC_CLR_FLG( a )                tkse_clr_flg( ctx->dec_flg_id, a )

# define GRL_BMPDEC_TER_TSK( a, b )             (b) = tkse_ter_tsk( a )

# define GRL_BMPDEC_EXT_TSK                     tkse_ext_tsk()
#endif/*BUILDENV_LINUX*/
#endif

/*******************************************************************************************
 *   Internal data type definitions
 *******************************************************************************************/

/**
 * List of resolver keys for one decoder.
 */
typedef struct GRL_BMPDEC_rslv_key_
{
    signed char       *draw_key;          /**< Resolver key for draw function             */
    signed char       *info_key;          /**< Resolver key for image info function       */
    signed char       *type_key;          /**< Resolver key for is type function          */
} GRL_BMPDEC_rslv_key;


/*******************************************************************************************
 *   Internal function prototypes (to be used only by the Decoder Core)
 *******************************************************************************************/


/**
 * Dummy function for decoding.
 * Is used for initializing the function lookup table.
 *
 * \param[in]   *info                   Decoding info
 *
 */
SVGError GRL_BMPDEC_draw_image_dummy( GRL_BMPDEC_dec_info *info );


/**
 * Dummy function for getting image info.
 * Is used for initializing the function lookup table.
 *
 * \param[in]   *ctx                    Context for Bitmap Decoder
 * \param[in]   *image                  Image definition structue
 * \param[in]   *image_info             Image info structure
 *
 */
SVGError GRL_BMPDEC_get_image_info_dummy( const SVGContextBmpDec   *ctx,
                                          const SVGImage     *image,
                                          SVGImageInfo       *image_info );


/**
 * Initialize function tables.
 * The function tables which store the format-specific
 * functions for decoding, getting image info etc. are
 * initialized to the dummy functions.
 *
 */
SVGError GRL_BMPDEC_init_tables( void );


/**
 * Register Decoder Modules.
 * Use the Resolver to look up the interface functions
 * of the available decoding modules. Each module should
 * have registered itself at the Resolver.
 *
 */
SVGError GRL_BMPDEC_reg_decoder( void );


/**
 * Create Decoder Context.
 * Create all resources required for a new decoder context
 * and add the context to the internal queue.
 *
 * \param       **new_ctx       Pointer to resulting context
 *
 */
SVGError GRL_BMPDEC_cre_ctx( SVGContextBmpDec **new_ctx );


/**
 * Destroy Decoder Context.
 * Destroy all resources that have been allocated for the context
 * and remove it from the internal queue.
 *
 * \param       *ctx           Pointer to context which should be destroyed
 *
 */
SVGError GRL_BMPDEC_dty_ctx( SVGContextBmpDec *ctx );


/**
 * Check application provided SVG Surface settings.
 *
 * \param[in]   *image                  Image definition structue
 *
 * \return      SVG_NO_ERROR            Structure is ok
 * \return      SVG_INVALID_VALUE       Structure is invalid
 *
 */
SVGError GRL_BMPDEC_check_mode_svg( const SVGImage *image );


/**
 * Check application provided Memory settings
 *
 * \param[in]   *image                  Image definition structue
 *
 * \return      SVG_NO_ERROR            Structure is ok
 * \return      SVG_INVALID_VALUE       Structure is invalid
 *
 */
SVGError GRL_BMPDEC_check_mode_mem( const SVGImage *image );


/**
 * Check basic application provided image settings.
 * This function is also used for checking in
 * svgGetImageInfo.
 *
 * \param[in]   *image                  Image definition structue
 *
 * \return      SVG_NO_ERROR            Structure is ok
 * \return      SVG_INVALID_VALUE       Structure is invalid
 *
 */
SVGError GRL_BMPDEC_check_img_basic( const SVGImage *image );


/**
 * Check application provided image settings.
 *
 * \param[in]   *image                  Image definition structue
 *
 * \return      SVG_NO_ERROR            Structure is ok
 * \return      SVG_INVALID_VALUE       Structure is invalid
 *
 */
SVGError GRL_BMPDEC_check_img( const SVGImage *image );


/**
 * Get info about application provided encoded image.
 * Uses the function table to call the correct
 * decoder-specific image info function.
 *
 * \param[in]   *ctx                    Context for Bitmap Decoder
 * \param[in]   *image                  Pointer to image definition structure
 * \param[out]  *image_info             Resulting image information
 *
 * \return      SVG_NO_ERROR            Information retrieval was successful
 * \return      SVG_INVALID_VALUE       Some parameter is incorrect or missing
 * \return      SVG_BMPDEC_DEC_ERROR    The decoder returned an error
 *
 * \see svgGetImageInfo
 */
SVGError GRL_BMPDEC_get_image_info( SVGContextBmpDec  *ctx,
                                    const SVGImage    *image,
                                    SVGImageInfo      *image_info );


/**
 * Get list of supported decoding formats
 * Uses the function table of registered draw functions to
 * detect which decoder modules are registered. The result
 * is returned as a bitmask.
 *
 * \param[out]  *formats                Resulting bitmask 
 *
 * \return      SVG_NO_ERROR            Format list creation was successful
 *
 * \see svgGetBmpDecFormats
 */
SVGError GRL_BMPDEC_get_formats( SVGBmpDecFormats       *formats );


/**
 * Decode the image synchronously.
 * Uses the function table to call the correct
 * decoder-specific image decoding function
 *
 * \param       *ctx                    Context for Bitmap Decoder
 *
 * \return      SVG_NO_ERROR            Decoding was successful
 * \return      SVG_OUT_OF_MEMORY       There was not enough memory for decoding
 * \return      SVG_INVALID_VALUE       Some parameter is incorrect or missing
 * \return      SVG_BMPDEC_INVALID_CLIP The clip rect is not valid
 * \return      SVG_BMPDEC_DEC_ERROR    The decoder returned an error
 *
 * \see svgFinishBmpDec
 */
SVGError GRL_BMPDEC_draw_image_sync ( SVGContextBmpDec     *ctx );


/**
 * Create decoder task.
 * Create task for parallel decoding (i.e. after svgFlushBmpDec) and
 * other required resources. The task will only be created during
 * the first flush to avoid unnecessary task creation for contexts
 * which only use svgFinishBmpDec.
 *
 * \param       *ctx                    Context for Bitmap Decoder
 *
 * \return      SVG_NO_ERROR                    Creation was successful
 * \return      SVG_BMPDEC_OS_ERROR             Could not create task
 *
 * \see svgFlushBmpDec
 */
SVGError GRL_BMPDEC_cre_dec_tsk( SVGContextBmpDec *ctx );


/**
 * Swap the two decoding queues.
 * Swap the write and the read queue of the given context.
 *
 * \param[in]   *ctx                    Context for Bitmap Decoder
 *
 */
void GRL_BMPDEC_swap_queues( SVGContextBmpDec *ctx );


/**
 * Decode the image asynchronously.
 * Uses the function table to call the correct
 * decoder-specific image decoding function. Create
 * a separate task for decoding as to return nearly immediately.
 *
 * \param[in]   *ctx                    Context for Bitmap Decoder
 *
 * \return      SVG_NO_ERROR            Decoding was successful
 * \return      SVG_OUT_OF_MEMORY       There was not enough memory for decoding
 * \return      SVG_INVALID_VALUE       Some parameter is incorrect or missing
 * \return      SVG_BMPDEC_INVALID_CLIP The clip rect is not valid
 * \return      SVG_BMPDEC_DEC_ERROR    The decoder returned an error
 *
 * \see svgFinishBmpDec
 */
SVGError GRL_BMPDEC_draw_image_async ( SVGContextBmpDec     *ctx );


/**
 * Wait for the asynchronous decoding to finish.
 * This function blocks until the currently running
 * decoding request is finished. If there is no
 * request currently being processes, the function returns
 * immediately.
 *
 * \param[in]   *ctx                    Context for Bitmap Decoder
 *
 * \return      SVG_NO_ERROR            Waiting was successful
 * \return      SVG_BMPDEC_OS_ERROR     An error occurred while accessing OS resources
 *
 * \see svgFlushBmpDec
 */
SVGError GRL_BMPDEC_wai_dec( SVGContextBmpDec *ctx );


/**
 * Get internal error code.
 * This function gets the internal error code either from
 * one context (if available) or globally.
 *
 * \param[in]   *ctx            Context for Bitmap Decoder
 *
 * \return      The internal error code
 * 
 * \see svgGetBmpDecError
 */
SVGError GRL_BMPDEC_get_error( SVGContextBmpDec  *ctx );

/**
 * Set internal error code.
 * This function sets the internal error code either for
 * one context (if available) or globally.
 *
 * \param[in]   *ctx            Context for Bitmap Decoder
 * \param[in]   error           Error code which will be set
 *
 * \see svgGetBmpDecError, GRL_BMPDEC_clr_error
 */
void GRL_BMPDEC_set_error( SVGContextBmpDec  *ctx,
                           SVGError          error );


/**
 * Clear internal error code.
 * This function sets the internal error code either for
 * one context (if available) or globally to SVG_NO_ERROR.
 *
 * \param[in]   *ctx            Context for Bitmap Decoder
 *
 * \see svgGetBmpDecError, GRL_BMPDEC_set_error
 */
void GRL_BMPDEC_clr_error(SVGContextBmpDec *ctx);


/**
 * Initialize context list.
 */
SVGError GRL_BMPDEC_clst_init( void );


/**
 * Uninitialize context list.
 */
SVGError GRL_BMPDEC_clst_uninit( void );


/**
 * Add context to context list.
 * 
 * \param[in]   *new_ctx        New context for adding
 * 
 * \return      SVG_BMPDEC_OS_ERROR     Access to lock failed
 * \return      SVG_BMPDEC_CTX_MAX      Maximum number of contexts reached
 */
SVGError GRL_BMPDEC_clst_add( SVGContextBmpDec *new_ctx );


/**
 * Remove context from context list.
 * 
 * \param[in]   *del_ctx        Context for removing
 *
 * \return      SVG_BMPDEC_OS_ERROR             Access to lock failed
 * \return      SVG_BMPDEC_INVALID_CONTEXT      A not-existing context was passed
 */
SVGError GRL_BMPDEC_clst_del( const SVGContextBmpDec *del_ctx );


/**
 * Get context with particular index.
 * 
 * \param[in]   ctx_idx         Index of context
 *
 * \return      Pointer to context
 */
SVGContextBmpDec* GRL_BMPDEC_clst_get_ctx( SVGUint32 ctx_idx );


/**
 * Check whether context is valid
 * 
 * \param[in]   ctx             Context for checking
 *
 * \return      SVG_TRUE if context is valid, SVG_FALSE otherwise
 */
SVGBoolean GRL_BMPDEC_clst_is_valid( const SVGContextBmpDec *ctx );


/**
 * Get number of registered contexts.
 * 
 * \return      Number of contexts
 */
SVGUint32 GRL_BMPDEC_clst_get_num( void );

#endif /* GRL_BITMAP_DECODER_CORE_H */
